/*
 * CombatCharacter.h
 *
 * Created 8/27/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/27/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef COMBAT_CHARACTER_H
 #define COMBAT_CHARACTER_H
 
 template <typename T> class CombatCharacter;
 
 #include "CharacterObject.h"
 
 #include "global.h"
 #include "DamageObjectCollection.h"
 #include "DamageObject.h"
 
 #include "CollisionHandlerEventNode.h"
 #include "StringConversion.h"
 
 /**
  * Class specification for CombatCharacter
  */
 template <typename T>
 class CombatCharacter : public CharacterObject<T>
 {
 // Data Members
 private:
    DamageObjectCollection<T> _dmg_objects;
 
 // Local Functions
 public:
    CombatCharacter( const T movement_speed = ONE );
    CombatCharacter( const CombatCharacter<T>& character );
    virtual ~CombatCharacter();
    inline CombatCharacter<T>& operator=( const CombatCharacter<T>& character );
    virtual inline void add_damage_object( DamageObject<T>* dmg_obj_Ptr );
    virtual inline void add_damage_object( const std::string& key, DamageObject<T>* dmg_obj_Ptr );
    virtual inline void detach_damage_object( const std::string& key );
    virtual inline void detach_damage_objects();
    virtual inline DamageObject<T>* get_damage_object( const std::string& key );
    virtual inline void remove_damage_object( const std::string& key );
    virtual inline void remove_damage_objects();
    //inline DamageObjectCollection<T>& get_damage_objects();
    
    // overloaded functions (NodePath)
    /*static void init_type() {
                                std::string template_type( typeid( T ).name() );
                                register_type(_type_handle, "CombatCharacter<" + template_type + ">" );
                            }*/
 
 // Protected Functions
 protected:
    
 // Static Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 CombatCharacter<T>::CombatCharacter( const T movement_speed )
                    : CharacterObject<T>( movement_speed )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 CombatCharacter<T>::CombatCharacter( const CombatCharacter<T>& character )
                    : CharacterObject<T>( character ),
                      _dmg_objects( character._dmg_objects )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 CombatCharacter<T>::~CombatCharacter()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified CombatCharacter to this CombatCharacter.
  *
  * @param (const CombatCharacter<T>&) character
  * @return CombatCharacter<T>&
  */
 template <typename T>
 inline CombatCharacter<T>& CombatCharacter<T>::operator=( const CombatCharacter<T>& character )
 {
    CharacterObject<T>::operator=( static_cast< CharacterObject<T> >( character ) );
    _dmg_objects = character._dmg_objects;
    
    return *this;
 }
 
 /**
  * add_damage_object() adds the DamageObject, specified by the dmg_obj_Ptr, using its name as the 
  * key, into the DamageObjectCollection for this CombatCharacter.
  *
  * @param (DamageObject<T>*) dmg_obj_Ptr
  */
 template <typename T>
 inline void CombatCharacter<T>::add_damage_object( DamageObject<T>* dmg_obj_Ptr )
 {
    nassertv( dmg_obj_Ptr != NULL );
    nassertv( !dmg_obj_Ptr->is_empty() );
    CombatCharacter<T>::add_damage_object( dmg_obj_Ptr->get_name(), dmg_obj_Ptr );
 }
 
 /**
  * add_damage_object() adds the DamageObject specified by the dmg_obj_Ptr with the specified key into the
  * DamageObjectCollection for this CombatCharacter.
  *
  * @param (const std::string&) key
  * @param (DamageObject<T>*) dmg_obj_Ptr
  */
 template <typename T>
 inline void CombatCharacter<T>::add_damage_object( const std::string& key, DamageObject<T>* dmg_obj_Ptr )
 {
    nassertv( dmg_obj_Ptr != NULL );
    _dmg_objects.add( key, dmg_obj_Ptr );
    
    // Tag the DamageObject as belonging to this Character
    std::string object_key( StringConversion::to_str( NodePath::get_key() ) );
    dmg_obj_Ptr->set_tag( OBJECT_KEY_TAG, object_key );
    
    // Tag all the Colliders already attached to the DamageObject as belonging to this Character
    dmg_obj_Ptr->CollisionObject<T>::tag_colliders( OBJECT_KEY_TAG, object_key );
 }
 
 /**
  * detach_damage_object() detaches the DamageObject matching the specified key
  * from its parent node, if the DamageObject has a parent node.
  *
  * @param (const std::string&) key
  */
 template <typename T>
 inline void CombatCharacter<T>::detach_damage_object( const std::string& key )
 {
    _dmg_objects.detach( key );
 }
 
 /**
  * detach_damage_objects() detaches all DamageObjects belonging to this Object
  * from their parents.
  */
 template <typename T>
 inline void CombatCharacter<T>::detach_damage_objects()
 {
    _dmg_objects.detach_all();
 }
 
 /**
  * get_damage_object() returns a pointer to the DamageObject matching the specified key.
  * If no DamageObject having the matching specified key is found, NULL is returned.
  *
  * @param (const std::string&) key
  * @return DamageObject<T>*
  */
 template <typename T>
 inline DamageObject<T>* CombatCharacter<T>::get_damage_object( const std::string& key )
 {
    return static_cast<DamageObject<T>*>( _dmg_objects.get( key ) );
 }
 
 /**
  * remove_damage_object() removes the DamageObject matching the specified key
  * from the DamageObjectCollection belonging to this Object.
  *
  * @param (const std::string&) key
  */
 template <typename T>
 inline void CombatCharacter<T>::remove_damage_object( const std::string& key )
 {
    _dmg_objects.remove( key );
 }
 
 /**
  * remove_damage_objects() removes all DamageObjects from the DamageObjectCollection
  * belonging to this Object.
  */
 template <typename T>
 inline void CombatCharacter<T>::remove_damage_objects()
 {
    _dmg_objects.clear();
 }
 
 /**
  * get_damage_objects() returns the DamageObjectCollection.
  *
  * @return DamageObjectCollection<T>&
  */
 //template <typename T>
 //inline DamageObjectCollection<T>& CombatCharacter<T>::get_damage_objects()
 //{
 //   return _dmg_objects;
 //}
 
 /** STATIC FUNCTIONS **/
 
 #endif // COMBAT_CHARACTER_H